# -*- mode: landisk-python; coding: utf-8; -*-
# vim:ts=4 sw=4 sts=4 ai si et sta

"""AmazonS3 File Access module.

Define classes for AmazonS3 file access.
"""

# common module
from __future__ import unicode_literals
from fileaccess import (FileObj, Access)
import os
import stat
import arrow
from buffer_object import BufferObject, BufferObjectW, ChunkedBuffer
from fileaccess_helper import *

# feature module
import re
from boto.s3.connection import S3Connection
from arrow.parser import ParserError as ArrowParseError
from buffer_object import BufferObjectR
from buffer_object2 import ChunkedBuffer
import threading
import sys
import boto.utils
from hashlib import md5


# TODO
import random
def get_random_string(length=30, chars='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'):
    return ''.join(random.choice(chars) for i in range(length))


# common constant
FILE_SEPARATOR = "/"
FILE_MODE = 0
DIR_MODE = stat.S_IFDIR
ERR_MSG_RENAME = "Invalid new name"
ERR_MSG_INVALID_FILE_LENGTH = "Invalid file length"
ERR_MSG_INVALID_OPEN_MODE = "Invalid open mode"
OPEN_MODE_READ = "r"
OPEN_MODE_WRITE = "w"


class AmazonS3Uploader(object):
    def __init__(self, key, access, length, reduced_redundancy, buffer_size):
        self.key = key
        self.access = access
        self.length = length
        self.reduced_redundancy = reduced_redundancy
        self.buffer_size = buffer_size
        self.part_num = 1
        self.buffer = None
        self.exception = None
        self.is_alive = True
        self.is_complete = False

    def close(self): pass

    def upload(self): pass


import functools
from threading import RLock
class _Sizemap(object):
    def __init__(self):
        self.__lock = RLock()
        self.__sizemap = dict()
    def get(self, fp):
        return self.__sizemap.get(fp, None)
    def set(self, fp, size):
        with self.__lock:
            self.__sizemap[fp] = size
    def unset(self, fp):
        with self.__lock:
            return self.__sizemap.pop(fp, None)
def _wrap_compute_hash(_o_compute_hash, _sizemap=_Sizemap()):
    @functools.wraps(_o_compute_hash)
    def _wrapped(fp, buf_size=8192, size=None, hash_algorithm=md5):
        _size = _sizemap.get(fp)
        if _size is None:
            return _o_compute_hash(fp,
                                   buf_size=buf_size,
                                   size=size,
                                   hash_algorithm=hash_algorithm)
        else:
            if size is not None:
                _size = size
            return _o_compute_hash(fp,
                                   buf_size=buf_size,
                                   size=_size,
                                   hash_algorithm=hash_algorithm)
            #return (None, None, size if size else _size)
    return _wrapped, _sizemap
boto.utils.compute_hash, _sizemap = _wrap_compute_hash(boto.utils.compute_hash)


class AmazonS3StandardUploader(AmazonS3Uploader):
    def __init__(self, key, access, length, reduced_redundancy, buffer_size):
        # def compute_hash_dummy(fp, buf_size=8192, size=None, hash_algorithm=md5):
        #     if size:
        #         data_size = size
        #     else:
        #         data_size = length
        #     return (None, None, data_size)
        # self.compute_hash_org = None
        # self.compute_hash_org = boto.utils.compute_hash
        # boto.utils.compute_hash = compute_hash_dummy

        super(AmazonS3StandardUploader, self).__init__(key, access, length, reduced_redundancy, buffer_size)
        self.buffer = ChunkedBuffer(size=self.length, buffer_size=self.buffer_size)
        _sizemap.set(self.buffer, length)
        self.thread = threading.Thread(target=self.upload)
        self.thread.start()

    def close(self):
        if self.is_alive:
            self.thread.join()
        _sizemap.unset(self.buffer)
        self.buffer.close()
        # if self.compute_hash_org:
        #     boto.utils.compute_hash = self.compute_hash_org
        if self.exception:
            raise self.exception[1], None, self.exception[2]

    def upload(self):
        try:
            self.buffer.set_r_mode()
            self.key.set_contents_from_file(self.buffer, reduced_redundancy=self.reduced_redundancy)
            self.is_complete = True
        except Exception as e:
            self.exception = sys.exc_info()
            self.buffer.close()
            self.is_alive = False

    def write(self, data):
        if self.is_alive:
            self.buffer.write(data)


class AmazonS3MultiPartUploader(AmazonS3Uploader):
    def __init__(self, key, access, length, reduced_redundancy, buffer_size):
        super(AmazonS3MultiPartUploader, self).__init__(key, access, length, reduced_redundancy, buffer_size)
        self.mp = access._bucket.initiate_multipart_upload(key.name, reduced_redundancy=self.reduced_redundancy)
        self.upload_size = 0
        self.buffer = BufferObjectW(size=buffer_size)

    def close(self):
        if self.is_complete:
            self.complete()
        else:
            self.cancel()
        self.buffer.close()

    def upload(self):
        with BufferObjectR(self.buffer.read()) as fp:
            fp.set_r_mode()
            self.mp.upload_part_from_file(fp, part_num=self.part_num)
        self.upload_size += self.buffer.length
        self.part_num += 1
        if self.upload_size == self.length:
            self.is_complete = True

    def write(self, data):
        r_buf = BufferObjectR(data=data)
        while r_buf.rest > 0:
            data = r_buf.read(self.buffer.lack)
            self.buffer.write(data)
            if self.buffer.lack == 0:
                self.upload()
                self.buffer.close()
                self.buffer = BufferObjectW(size=min(self.buffer_size, self.length - self.upload_size))
        r_buf.close()

    def complete(self):
        self.mp.complete_upload()
        self.buffer.close()

    def cancel(self):
        self.mp.cancel_upload()
        self.buffer.close()


class AmazonS3FileObj(FileObj):
    """Class for AmazonS3 file."""

    def __init__(self, access, path, mode, length, chunk_size, do_use_multipart_upload, reduced_redundancy):
        """Constructor."""

        super(AmazonS3FileObj, self).__init__()
        self._do_use_multipart_upload = do_use_multipart_upload
        self._reduced_redundancy = reduced_redundancy
        self._access = access
        self._path = path
        self._mode = (filter((lambda m: m in mode.lower()), (OPEN_MODE_READ, OPEN_MODE_WRITE)) or [None])[0]
        self._length = length
        self._chunk_size = chunk_size
        if self._chunk_size is None:
            self._chunk_size = 8 * 1024 * 1024
        self._r_pos = 0
        self._w_pos = 0
        self._downloader = None
        self._uploader = None
        self._buffer = None

        if self._mode == OPEN_MODE_READ:
            self._length = self._access.stat_file(self._path)[stat.ST_SIZE]
            self._download()
        elif self._mode == OPEN_MODE_WRITE:
            if self._length < 0:
                raise ValueError(ERR_MSG_INVALID_FILE_LENGTH)
            self._upload()
        else:
            raise ValueError(ERR_MSG_INVALID_OPEN_MODE)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()
        return False

    def close(self):
        """Close file."""

        try:
            if self._mode == OPEN_MODE_WRITE:
                if self._uploader:
                    self._uploader.close()
        finally:
            if self._buffer:
                self._buffer.close()

    def read(self, size=-1):
        """Return a "size" bytes data from current position in file.
        If the size argument is negative or omitted, then return all data.
        """

        if size < 0:
            read_size = self._length
        else:
            read_size = min(size, self._length - self._r_pos)

        ret_buf = BufferObjectW(size=read_size)
        while ret_buf.lack > 0:
            ret_buf.write(self._buffer.read(ret_buf.lack))
            if self._buffer.rest == 0:
                self._buffer.close()
                self._download()

        self._r_pos += ret_buf.length
        ret = ret_buf.read()
        ret_buf.close()

        return ret

    def _download(self):
        """Download to buffer"""

        if not self._downloader:
            key = self._access._get_file(self._path, be_raise=True)
            if key is not None:
                self._length = self._access._stat(key)[stat.ST_SIZE]
                self._downloader = iter(key)

        data = "".encode("utf-8")
        try:
            data = self._downloader.next()
        except StopIteration:
            pass
        self._buffer = BufferObjectR(data)

    def write(self, data):
        """Write a "data" data to current position in file."""
        self._uploader.write(data)

    def _upload(self):
        if not self._uploader:
            key = self._access._get_file(self._path)
            if key is None:
                key = self._access._bucket.new_key(self._path)

            if self._length >= self._chunk_size and self._do_use_multipart_upload:
                self._uploader = AmazonS3MultiPartUploader(
                    key=key,
                    access=self._access,
                    length=self._length,
                    reduced_redundancy=self._reduced_redundancy,
                    buffer_size=self._chunk_size,
                )
            else:
                self._uploader = AmazonS3StandardUploader(
                    key=key,
                    access=self._access,
                    length=self._length,
                    reduced_redundancy=self._reduced_redundancy,
                    buffer_size=self._chunk_size,
                )


class AmazonS3Access(Access):
    """class accessing to AmazonS3 file."""

    def __init__(self,
                 accesskey=None,
                 secretkey=None,
                 bucket=None,
                 host=None,
                 port=None,
                 is_secure=True,
                 reduced_redundancy=None,
                 do_use_multipart_upload=True,
                 proxy=None,
                 proxy_port=None,
                 proxy_user=None,
                 proxy_pass=None,
                 ):
        """Constructor.

        Args:
            accesskey (unicode): client access key
            secretkey (unicode): client access secret
            host (unicode): The host to make the connection to
            port (int): The port to use to connect
            is_secure (bool): Whether the connection is over SSL
            bucket (unicode): The name of the bucket
            reduced_redundancy (bool):
                In multipart uploads, the storage class is specified when initiating the upload,
                not when uploading individual parts.
                So if you want the resulting key to use the reduced redundancy storage class set
                this flag when you initiate the upload.
            do_use_multipart_upload (bool):
                 whether to use multipart uploads
            proxy (unicode): Address/hostname for a proxy server
                If None, bypass proxies
            proxy_port (int): The port to use when connecting over a proxy
            proxy_user (unicode): The username to connect with on the proxy
            proxy_pass (unicode): The password to use when connection over a proxy
        """

        super(AmazonS3Access, self).__init__()

        self._reduced_redundancy = reduced_redundancy
        self._do_use_multipart_upload = do_use_multipart_upload

        self._conn = S3Connection(
            host=host,
            port=port,
            is_secure=is_secure,
            aws_access_key_id=accesskey,
            aws_secret_access_key=secretkey,
            proxy=proxy,
            proxy_port=proxy_port,
            proxy_user=proxy_user,
            proxy_pass=proxy_pass,
        )
        self._bucket = None
        if bucket:
            self._bucket = self._conn.get_bucket(bucket)

            self._do_use_multipart_upload = do_use_multipart_upload and self.can_multipart_upload

    @property
    def use_multipart_upload(self):
        return self._do_use_multipart_upload

    @property
    def can_multipart_upload(self):

        # Assume multipart_upload is possible.
        return True

        can = getattr(self, '__can_multipart_upload', None)
        if can is None:
            can = self._ask_can_multipart_upload(self._reduced_redundancy)
            setattr(self, '__can_multipart_upload', can)
        return can

    def _ask_can_multipart_upload(self, reduced_redundancy=False):
        """
        マルチパートアップロード可能かテストする
        パートサイズの制限: 5MB~5GB
        """
        try:
            path = get_random_string(length=50)
            chunk_size = 5 * 1024 * 1024  # MB
            length = chunk_size * 2
            with AmazonS3FileObj(access=self,
                path=path,
                mode=OPEN_MODE_WRITE,
                length=length,
                chunk_size=chunk_size,
                do_use_multipart_upload=True,
                reduced_redundancy=reduced_redundancy,
            ) as f:
                pass
                f.write(os.urandom(length))
            self.delete_file(path)
            return True
        except Exception as err:
            return False

    @property
    def max_num_transfer_threads(self):
        """Return maximum thread number for file transfer."""

        try:
            if self.use_multipart_upload:
                return 5
            return 1
        except AttributeError as err:
            return 1

    @property
    def auth_url(self):
        return None

    def authenticate(self, params=None):
        pass

    @staticmethod
    def _get_file_name(file_path):
        """Get path of "file" for S3

        Args:
            file_path (unicode): path
        Returns:
            string:
        """
        return file_path.strip(FILE_SEPARATOR)

    @staticmethod
    def _get_directory_name(file_path):
        """Get path of "directory" for S3

        Args:
            file_path (unicode): path
        Returns:
            string:
        """
        return file_path.strip(FILE_SEPARATOR) + FILE_SEPARATOR

    def _get_file(self, file_path, be_raise=False):
        """Get file object

        Args:
            file_path (unicode): path
            be_raise (bool): If the item is not found,
                an exception is raised if is True,
                or return None if is False
        Returns:
            boto.S3.key.Key if found, None if not Found
        """
        _target_path = self._get_file_name(file_path)
        item = self._bucket.get_key(_target_path)
        if item is None and be_raise:
            raise LookupError(_target_path)
        return item

    def _get_directory(self, file_path, be_raise=False):
        """Get directory object

        Args:
            file_path (unicode): path
            be_raise (bool): If the item is not found,
                an exception is raised if is True,
                or return None if is False
        Returns:
            boto.S3.key.Key if found, None if not Found
        """
        _target_path = self._get_directory_name(file_path)
        item = self._bucket.get_key(_target_path)
        if item is None and be_raise:
            raise LookupError(_target_path)
        return item

    @staticmethod
    def _stat(target_item):
        """Return a stat information dictionary for "target_item".

        Args:
            target_item (boto.s3.key.Key): target item
        Returns:
            dict: stat of item

            {
                stat.ST_MODE: mode,
                stat.ST_MTIME: mtime,
                stat.ST_CTIME: ctime,
                stat.ST_SIZE: size,
            }

                {
                    stat.ST_MODE: 0 if target is file, stat.S_IFDIR if target is directory,

                    stat.ST_MTIME: modified time for target,

                    stat.ST_CTIME: changed time for target, but S3 is not support,

                    stat.ST_SIZE: file size if target is file, 0 if target is directory,

                }
        """

        if target_item.name[-1] == FILE_SEPARATOR:
            _mode = DIR_MODE
        else:
            _mode = FILE_MODE

        _mtime = target_item.last_modified
        if _mtime is not None:
            try:
                _mtime = arrow.get(_mtime).timestamp
            except ArrowParseError:
                try:
                    _mtime = arrow.get(_mtime, "ddd, DD MMM YYYY HH:mm:ss ZZZ").timestamp
                except ArrowParseError:
                    _mtime = arrow.get(_mtime, "ddd, DD MMM YYYY HH:mm:ss").replace(tzinfo="+09:00").timestamp

        _ctime = None
        _size = None
        if _mode == FILE_MODE:
            _size = target_item.size

        return {
            stat.ST_MODE: _mode,
            stat.ST_MTIME: _mtime,
            stat.ST_CTIME: _ctime,
            stat.ST_SIZE: _size,
        }

    def make_directory(self, dir_path):
        """Create "dir_path" directory."""

        if dir_path == "" or dir_path == FILE_SEPARATOR:
            # if blank, do nothing
            return

        target_id = self._get_directory(dir_path)
        if target_id is not None:
            # if exists, do nothing
            return

        parent_path = self._get_directory_name(
            os.path.dirname(
                self._get_file_name(dir_path)))
        if parent_path == FILE_SEPARATOR or parent_path == "":
            pass
        else:
            parent_item = self._get_directory(parent_path)
            if parent_item is None:
                self.make_directory(parent_path)

        new_dir_path = self._get_directory_name(dir_path)
        key = self._bucket.new_key(new_dir_path)
        key.set_contents_from_string("")

    def delete_directory(self, dir_path):
        """Delete "dir_path" directory."""

        prefix = self._get_directory_name(dir_path)
        pattern = re.compile(r"^%s" % re.escape(prefix))

        try:
            item_list = self._bucket.list(prefix=prefix)
            for item in item_list:
                name = item.name
                if re.match(pattern, name):
                    item.delete()
        except:
            raise LookupError(dir_path)

    def stat_directory(self, dir_path):
        """Return a stat information dictionary for "file_path"
           which keys below:
            stat.ST_MODE, stat.ST_MTIME, stat.CTIME, stat.ST_SIZE
        """

        class _DummyKey(object):
            def __init__(self, **kwargs):
                for k, v in kwargs.iteritems():
                     setattr(self, k, v)
        try:
            k = self._get_directory(dir_path, be_raise=True)
            s = self._stat(k)
            return s
        except LookupError as err:
            prefix = self._get_directory_name(dir_path)
            for k in self._bucket.get_all_keys(prefix=prefix, max_keys=1):
                s = self._stat(_DummyKey(name=prefix, last_modified=0))
                return s
            else:
                raise

    def rename_directory(self, dir_path, new_name):
        """Rename a name of "dir_path" to "new_name"."""

        if FILE_SEPARATOR in new_name:
            raise Exception(ERR_MSG_RENAME, {"new_name": new_name})

        src_dir = self._get_directory(dir_path, be_raise=True)

        prefix = self._get_directory_name(src_dir.name)
        pattern = re.compile(r"^%s" % re.escape(prefix))
        repl = "%s%s" % (
            self._get_directory_name(
                os.path.dirname(
                    self._get_file_name(src_dir.name))),
            self._get_directory_name(new_name))

        item_list = self._bucket.list(prefix=prefix)
        for item in item_list:
            new_name = re.sub(pattern, repl, item.name)
            # print item.name
            # print "-> " + new_name
            item.copy(item.bucket.name, new_name, reduced_redundancy=self._reduced_redundancy)
            item.delete()

    def move_directory(self, dir_path, new_dir_path):
        """Move "dir_path" to "new_path"."""

        src_dir = self._get_directory(dir_path, True)
        dest_dir = self._get_directory(new_dir_path, True)

        prefix = self._get_directory_name(src_dir.name)
        pattern = re.compile(r"^%s" % re.escape(prefix))
        repl = "%s%s" % (
            self._get_directory_name(dest_dir.name),
            self._get_directory_name(
                os.path.basename(
                    self._get_file_name(src_dir.name))))

        item_list = self._bucket.list(prefix=prefix)
        for item in item_list:
            new_name = re.sub(pattern, repl, item.name)
            # print item.name
            # print "-> " + new_name
            item.copy(item.bucket.name, new_name, reduced_redundancy=self._reduced_redundancy)
            item.delete()

    def get_contents(self, dir_path, with_stat=False):
        """Return a contents dictionary in "dir_path".
           A stat information must be included, if "with_stat" is True, 
        """

        contents = {}
        prefix = self._get_directory_name(dir_path)
        if prefix == "/":
            item_list = self._bucket.list(delimiter=FILE_SEPARATOR)
            for item in item_list:
                name = item.name
                splitted = dict(enumerate(name.split("/")))
                if splitted.get(1) == "" or splitted.get(1) is None:
                    _name = os_path_bottom_name(name)
                    contents[_name] = {}
        else:
            pattern = re.compile(r"^%s" % re.escape(prefix))
            item_list = self._bucket.list(prefix=prefix,
                                          delimiter=FILE_SEPARATOR)
            found = False
            for item in item_list:
                # この self._bucket.list であれば、
                # 例え空フォルダであっても、
                # フォルダ自体のオブジェクトが list されるため、
                # ここに入った時点でフォルダが存在することはわかる
                found = True
                name = item.name
                base_name = self._get_file_name(re.sub(pattern, "", name))
                if re.match(pattern, name) \
                        and FILE_SEPARATOR not in base_name\
                        and base_name != "":
                    _name = os_path_bottom_name(name)
                    contents[_name] = {}
            if not found:
                raise LookupError(dir_path)
        return contents

        # before
        contents = {}
        _dir = self._get_directory(dir_path, be_raise=True)
        if _dir:
            prefix = self._get_directory_name(dir_path)
            if prefix == "/":
                item_list = self._bucket.list(delimiter=FILE_SEPARATOR)
                for item in item_list:
                    name = item.name
                    splitted = dict(enumerate(name.split("/")))
                    if splitted.get(1) == "" or splitted.get(1) is None:
                        _name = os_path_bottom_name(name)
                        contents[_name] = {}
            else:
                pattern = re.compile(r"^%s" % re.escape(prefix))
                item_list = self._bucket.list(prefix=prefix,
                                                delimiter=FILE_SEPARATOR)
                for item in item_list:
                    name = item.name
                    base_name = self._get_file_name(re.sub(pattern, "", name))
                    if re.match(pattern, name) \
                            and FILE_SEPARATOR not in base_name\
                            and base_name != "":
                        _name = os_path_bottom_name(name)
                        contents[_name] = {}
        return contents

    def get_updated_contents(self, dir_path):
        """Return a updated contents dictionary in "dir_path"."""

        raise NotImplementedError

    def open(self, file_path, mode="r", length=-1, chunk_size=None):
        """Return a class instance of the "FileObj" or sub class
           corresponding to "file_path".
        """

        return AmazonS3FileObj(
            access=self,
            path=self._get_file_name(file_path),
            mode=mode,
            length=length,
            chunk_size=chunk_size,
            do_use_multipart_upload=self._do_use_multipart_upload,
            reduced_redundancy=self._reduced_redundancy,
        )

    def delete_file(self, file_path):
        """Delete "file_path" file."""

        item = self._get_file(file_path, be_raise=True)
        item.delete()

    def stat_file(self, file_path):
        """Return a stat information dictionary for "file_path"
           which keys below:
            stat.ST_MODE, stat.ST_MTIME, stat.CTIME, stat.ST_SIZE
        """

        item = self._get_file(file_path, be_raise=True)
        return self._stat(item)

    def rename_file(self, file_path, new_name):
        """Rename a name of "file_path" to "new_name"."""

        if FILE_SEPARATOR in new_name:
            raise Exception("Invalid new_name", {"new_name": new_name})

        src_item_name = self._get_file_name(file_path)
        dir_name = os.path.dirname(src_item_name)
        dest_item_name = os_path_join(dir_name, new_name)

        src_item = self._get_file(src_item_name, be_raise=True)
        src_item.copy(src_item.bucket.name, dest_item_name)
        src_item.delete()

    def move_file(self, file_path, new_dir_path):
        """Move "file_path" to "new_path"."""

        src_item_name = self._get_file_name(file_path)
        base_name = os.path.basename(src_item_name)
        dest_item_name = os_path_join(new_dir_path, base_name)

        src_item = self._get_file(src_item_name, be_raise=True)
        src_item.copy(src_item.bucket.name, dest_item_name)
        src_item.delete()

    def test(self):
        pass
